const CACHE_NAME = 'phaset-cache-v1';
const CACHE_MAX_AGE = 30;
const CACHEABLE_PATHS = [
  '/api/organization*',
  '/api/tenant*',
  '/api/catalog*',
  '/api/record*',
  '/api/dora*',
  '/api/health*'
];

async function log(message) {
  const cache = await caches.open(CACHE_NAME);
  const logEnabled = await cache.match('logEnabled');

  if (!logEnabled) {
    return;
  }

  console.log(`[Service Worker] ${message}`);
}

self.addEventListener('install', (event) => {
  event.waitUntil(
    caches.open(CACHE_NAME).then(() => {
      self.skipWaiting(); // Force the waiting service worker to become the active service worker
    })
  );
});

self.addEventListener('activate', (event) => {
  event.waitUntil(
    caches
      .keys()
      .then((cacheNames) => {
        return Promise.all(
          cacheNames.map((cacheName) => {
            if (cacheName !== CACHE_NAME) return caches.delete(cacheName);
            return null;
          })
        );
      })
      .then(() => {
        self.clients.claim(); // Ensure that the service worker takes control immediately
      })
  );
});

const pendingRequests = new Map();

self.addEventListener('fetch', (event) => {
  // Only handle HTTP/HTTPS requests
  if (
    !event.request.url.startsWith('http') ||
    ['GET', 'OPTIONS'].includes(event.request.method) === false ||
    !CACHEABLE_PATHS.some((path) => event.request.url.match(path))
  ) {
    return;
  }

  const fetchAndCache = async () => {
    const cache = await caches.open(CACHE_NAME);
    const cachedResponse = await cache.match(event.request.url);

    // Check if the response is in the cache and is not stale
    if (cachedResponse) {
      if (
        (Date.now() -
          parseInt(cachedResponse.headers.get('cache-date') ?? '0', 10)) /
          1000 <
        CACHE_MAX_AGE
      ) {
        log(`Cache hit for ${event.request.url}`);
        return cachedResponse;
      } else {
        log(
          `Cache hit for ${event.request.url} but it's stale (${cachedResponse.headers.get('cache-date')})`
        );
      }
    }

    log(`Cache miss for ${event.request.url}`);
    const response = await fetch(event.request);

    // Return response without cache if it's not ok response
    if (!response.ok) {
      return response;
    }

    // Create a new response with the 'Date' header set to the current date, for cache validation
    const headers = {};
    response.headers.forEach((value, name) => {
      headers[name] = value;
    });
    headers['cache-date'] = Date.now();
    const responseWithDate = new Response(response.body, {
      status: response.status,
      statusText: response.statusText,
      headers: headers
    });
    cache.put(event.request.url, responseWithDate.clone());

    return responseWithDate;
  };

  let promise;

  if (pendingRequests.has(event.request.url)) {
    log(`Request for ${event.request.url} is already pending`);
    promise = (async () => {
      await pendingRequests.get(event.request.url);
      return fetchAndCache();
    })();
  } else {
    promise = fetchAndCache().finally(() => {
      log(`Request for ${event.request.url} is done, removing from pending`);
      pendingRequests.delete(event.request.url);
    });
    log(`Request for ${event.request.url} is not pending, adding to pending`);
    pendingRequests.set(event.request.url, promise);
  }

  event.respondWith(promise);
});

self.addEventListener('message', async (event) => {
  if (event.data && event.data.type === 'invalidate-cache') {
    const cache = await caches.open(CACHE_NAME);
    const keys = await cache.keys();
    keys.forEach((request) => {
      if (request.url.match(event.data.url)) {
        cache.delete(request);
      }
    });
  } else if (event.data && event.data.type === 'enable-log') {
    const cache = await caches.open(CACHE_NAME);
    cache.put('logEnabled', new Response('true'));
  }
});
